<?php

namespace App\Http\Controllers;

use App\Models\Masarif;
use App\Models\Post;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth; // <-- هاد السطر مهم

class MasarifController extends Controller
{
    // INDEX – عرض اللائحة


public function index(Request $request)
{
    $userId = Auth::id();

    // Base query: masarif dyal user connecté
    $query = Masarif::where('user_id', $userId);

    // Inputs
    $periode = $request->input('periode');
    $date_debut = $request->input('date_debut');
    $date_fin   = $request->input('date_fin');

    // Filtrage selon période
    if ($periode) {
        switch ($periode) {
            case 'aujourdhui':
                $query->whereDate('date', now()->toDateString());
                break;

            case 'cette_semaine':
                $query->whereBetween('date', [now()->startOfWeek(), now()->endOfWeek()]);
                break;

            case 'ce_mois':
                $query->whereMonth('date', now()->month)
                      ->whereYear('date', now()->year);
                break;

            case 'cette_annee':
                $query->whereYear('date', now()->year);
                break;

            case 'intervalle':
                if ($date_debut && $date_fin) {
                    $query->whereBetween('date', [$date_debut, $date_fin]);
                }
                break;
        }
    }

    // Total montant après filtration
    $totalMontant = $query->sum('montant');

    // Pagination
    $masarif = $query->orderBy('date', 'desc')->paginate(10)->withQueryString();

    return view('masarif.index', compact('masarif', 'totalMontant'));
}



    // CREATE – صفحة إضافة مصاريف ل Post معين
public function create(Post $post)
{
    $user = Auth::user(); // المستخدم المتصل
    return view('masarif.create', compact('post','user'));
}

    // STORE – تسجل مصروف جديد مرتبط ب post + user
public function store(Request $request)
{
    $request->validate([
        'titre'       => 'required',
        'montant'     => 'required|numeric',
        'date'        => 'required|date',
        'observation' => 'required',
        'post_id'     => 'required|exists:posts,id'
    ]);

    Masarif::create([
        'titre'       => $request->titre,
        'montant'     => $request->montant,
        'date'        => $request->date,
        'observation' => $request->observation,
        'post_id'     => $request->post_id,
        'user_id'     => Auth::id(), // <--- هذا هو المهم
    ]);

    return redirect()->route('masarif.index');
}


    // SHOW
    public function show($id)
    {
        $masarif = Masarif::findOrFail($id);
        return view('masarif.show', compact('masarif'));
    }

    // EDIT
    public function edit($id)
    {
        $masarif = Masarif::findOrFail($id);
        return view('masarif.edit', compact('masarif'));
    }

    // UPDATE
    public function update(Request $request, $id)
    {
        $masarif = Masarif::findOrFail($id);

        $request->validate([
            'titre'       => 'required',
            'montant'     => 'required|numeric',
            'date'        => 'required|date',
            'observation' => 'required',
        ]);

        $masarif->update($request->all());

        return redirect()->route('masarif.index');
    }

    // DELETE
    public function destroy($id)
    {
        Masarif::findOrFail($id)->delete();

        return redirect()->route('masarif.index');
    }

public function profile()
{
    $user = Auth::user();

    // جميع posts ديالو
    $posts = $user->posts; // خص يكون عندك relation posts ف User model

    return view('masarif.profile', compact('user','posts'));
}

}
